function openTab(evt, tabName) {
    var i, tabcontent, tablinks;
    tabcontent = document.getElementsByClassName("tab-content");
    for (i = 0; i < tabcontent.length; i++) {
        tabcontent[i].style.display = "none";
    }
    tablinks = document.getElementsByClassName("tab-link");
    for (i = 0; i < tablinks.length; i++) {
        tablinks[i].classList.remove("active");
    }
    document.getElementById(tabName).style.display = "block";
    evt.currentTarget.classList.add("active");

    // If the RPM tab is opened for the first time, render the charts.
    if (tabName === 'rpm' && !vitalsApp.chartsRendered) {
        // Use a timeout to ensure the canvas is visible before rendering.
        setTimeout(() => {
            // Default to 7-day range for RPM charts
            vitalsApp.renderCharts('7');
            vitalsApp.chartsRendered = true;
        }, 50);
    }
}

function generateReport() {
    alert("Generating report...");
}

function exportData() {
    alert("Exporting data...");
}

function updatePatientSummary(patient) {
    const summaryContainer = document.getElementById('patient-summary');
    if (!summaryContainer || !patient) return;

    // Using data from vitalsApp for the summary
    const latestVitals = vitalsApp.data['7'];
    const lastHr = latestVitals.hr.slice(-1)[0];
    const lastBpSys = latestVitals.bp_sys.slice(-1)[0];
    const lastBpDia = latestVitals.bp_dia.slice(-1)[0];
    const lastSpo2 = latestVitals.spo2.slice(-1)[0];

    const summaryData = {
        medicalInfo: `<strong>Vitals:</strong> HR: ${lastHr}, BP: ${lastBpSys}/${lastBpDia}, SpO2: ${lastSpo2}%<br>
                      <strong>Conditions:</strong> Hypertension, Type 2 Diabetes`,
        recentActivity: `<strong>Last Check-in:</strong> ${patient.lastCheckin || 'N/A'}<br>
                         <strong>Last Alert:</strong> High BP at 3:00 PM`,
        carePlan: `<strong>Plan:</strong> Low-sodium diet, daily exercise.<br>
                   <strong>Progress:</strong> ${patient.progress || 0}%`,
        alerts: `<div class="alert warning">High BP detected at 3:00 PM</div>`
    };

    document.getElementById('summary-medical-info').innerHTML = summaryData.medicalInfo;
    document.getElementById('summary-recent-activity').innerHTML = summaryData.recentActivity;
    document.getElementById('summary-care-plan').innerHTML = summaryData.carePlan;
    document.getElementById('summary-alerts').innerHTML = summaryData.alerts;
}

const vitalsApp = {
    charts: {
        hr: null,
        bp: null,
        spo2: null,
        weight: null,
    },
    chartsRendered: false,
    data: {
        '7': { // Last 7 days
            labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
            hr: [72, 75, 71, 78, 105, 73, 76], // Abnormal HR
            bp_sys: [120, 122, 118, 125, 135, 121, 123], // Abnormal BP
            bp_dia: [80, 81, 79, 82, 88, 80, 81], // Abnormal BP
            spo2: [98, 97, 99, 96, 94, 98, 97], // Abnormal SpO2
            weight: [180, 181, 180.5, 181.2, 180.8, 181.5, 181.3]
        },
        '30': { // Last 30 days
            labels: Array.from({ length: 30 }, (_, i) => `Day ${i + 1}`),
            hr: Array.from({ length: 30 }, () => Math.floor(Math.random() * 30) + 60),
            bp_sys: Array.from({ length: 30 }, () => Math.floor(Math.random() * 20) + 110),
            bp_dia: Array.from({ length: 30 }, () => Math.floor(Math.random() * 15) + 75),
            spo2: Array.from({ length: 30 }, () => Math.floor(Math.random() * 5) + 95),
            weight: Array.from({ length: 30 }, () => parseFloat((180 + (Math.random() * 5 - 2.5)).toFixed(1)))
        },
        'all': { // All time
            labels: Array.from({ length: 90 }, (_, i) => `Day ${i + 1}`),
            hr: Array.from({ length: 90 }, () => Math.floor(Math.random() * 30) + 60),
            bp_sys: Array.from({ length: 90 }, () => Math.floor(Math.random() * 20) + 110),
            bp_dia: Array.from({ length: 90 }, () => Math.floor(Math.random() * 15) + 75),
            spo2: Array.from({ length: 90 }, () => Math.floor(Math.random() * 5) + 95),
            weight: Array.from({ length: 90 }, () => parseFloat((180 + (Math.random() * 10 - 5)).toFixed(1)))
        }
    },

    getPointBackgroundColor: function(value, normalMin, normalMax) {
        return value < normalMin || value > normalMax ? 'red' : 'rgba(75, 192, 192, 1)';
    },

    renderCharts: function(range = '7') {
        const data = this.data[range];

        if (this.charts.hr) this.charts.hr.destroy();
        if (this.charts.bp) this.charts.bp.destroy();
        if (this.charts.spo2) this.charts.spo2.destroy();
        if (this.charts.weight) this.charts.weight.destroy();

        // Heart Rate Chart
        const hrCtx = document.getElementById('hr-chart').getContext('2d');
        this.charts.hr = new Chart(hrCtx, {
            type: 'line',
            data: {
                labels: data.labels,
                datasets: [{
                    label: 'Heart Rate',
                    data: data.hr,
                    borderColor: 'rgba(255, 99, 132, 1)',
                    backgroundColor: data.hr.map(v => this.getPointBackgroundColor(v, 60, 100)),
                    borderWidth: 2,
                    pointRadius: 5,
                    pointHoverRadius: 7
                }]
            }
        });

        // Blood Pressure Chart
        const bpCtx = document.getElementById('bp-chart').getContext('2d');
        this.charts.bp = new Chart(bpCtx, {
            type: 'line',
            data: {
                labels: data.labels,
                datasets: [
                    {
                        label: 'Systolic',
                        data: data.bp_sys,
                        borderColor: 'rgba(54, 162, 235, 1)',
                        backgroundColor: data.bp_sys.map(v => this.getPointBackgroundColor(v, 90, 130)),
                        borderWidth: 2,
                        pointRadius: 5,
                        pointHoverRadius: 7
                    },
                    {
                        label: 'Diastolic',
                        data: data.bp_dia,
                        borderColor: 'rgba(75, 192, 192, 1)',
                        backgroundColor: data.bp_dia.map(v => this.getPointBackgroundColor(v, 60, 85)),
                        borderWidth: 2,
                        pointRadius: 5,
                        pointHoverRadius: 7
                    }
                ]
            }
        });

        // SpO2 Chart
        const spo2Ctx = document.getElementById('spo2-chart').getContext('2d');
        this.charts.spo2 = new Chart(spo2Ctx, {
            type: 'line',
            data: {
                labels: data.labels,
                datasets: [{
                    label: 'SpO2',
                    data: data.spo2,
                    borderColor: 'rgba(153, 102, 255, 1)',
                    backgroundColor: data.spo2.map(v => this.getPointBackgroundColor(v, 95, 100)),
                    borderWidth: 2,
                    pointRadius: 5,
                    pointHoverRadius: 7
                }]
            },
            options: {
                scales: {
                    y: {
                        min: 90,
                        max: 100
                    }
                }
            }
        });

        // Weight Chart
        const weightCtx = document.getElementById('weight-chart').getContext('2d');
        this.charts.weight = new Chart(weightCtx, {
            type: 'line',
            data: {
                labels: data.labels,
                datasets: [{
                    label: 'Weight',
                    data: data.weight,
                    borderColor: 'rgba(255, 159, 64, 1)',
                    borderWidth: 2,
                    pointRadius: 5,
                    pointHoverRadius: 7
                }]
            }
        });
    }
};

document.addEventListener('DOMContentLoaded', () => {
    const themeSelect = document.getElementById('theme-select');
    const body = document.body;

    const applyTheme = (theme) => {
        body.className = theme;
        localStorage.setItem('selectedTheme', theme);
    };

    const savedTheme = localStorage.getItem('selectedTheme');
    if (savedTheme) {
        body.className = savedTheme;
        if (themeSelect) {
            themeSelect.value = savedTheme;
        }
    } else {
        applyTheme('purple');
        if (themeSelect) {
            themeSelect.value = 'purple';
        }
    }

    if (themeSelect) {
        themeSelect.addEventListener('change', () => {
            applyTheme(themeSelect.value);
        });
    }

    // Set user name in the header
    const userNameSpanPatient = document.getElementById('user-name-patient');
    if (userNameSpanPatient) {
        // In a real application, you would get this from a login system.
        userNameSpanPatient.textContent = 'Milan Nesic'; 
    }
    
    // Detailed patient info, which could come from a separate API call in a real app
    const detailedPatients = {
        '123456': { name: 'John Doe', dob: '01/15/1960', gender: 'Male', contact: '(123) 456-7890', address: 'john.doe@example.com', preferredLanguage: 'English', assignedNurse: 'Jane Doe' },
        '789012': { name: 'Jane Smith', dob: '05/22/1975', gender: 'Female', contact: '(123) 555-0101', address: 'jane.smith@example.com', preferredLanguage: 'Spanish', assignedNurse: 'John Smith' },
        '345678': { name: 'Bob Johnson', dob: '11/10/1960', gender: 'Male', contact: '(123) 555-0102', address: 'bob.johnson@example.com', preferredLanguage: 'English', assignedNurse: 'Jane Doe' },
        '654321': { name: 'Sarah Connor', dob: '05/05/1965', gender: 'Female', contact: '(123) 555-0106', address: 'sarah.connor@example.com', preferredLanguage: 'English', assignedNurse: 'John Smith' },
        '901234': { name: 'Alice Williams', dob: '03/15/1990', gender: 'Female', contact: '(123) 555-0104', address: 'alice.williams@example.com', preferredLanguage: 'French', assignedNurse: 'Emily White' },
        '567890': { name: 'Charlie Brown', dob: '08/20/2005', gender: 'Male', contact: '(123) 555-0105', address: 'charlie.brown@example.com', preferredLanguage: 'English', assignedNurse: 'Emily White' },
    };

    function loadPatientData() {
        const params = new URLSearchParams(window.location.search);
        const patientId = params.get('patientId') || '123456'; // Default to John Doe

        // Get detailed info from the local object
        const patientDetails = detailedPatients[patientId];

        if (!patientDetails) {
            document.getElementById('patient-name').textContent = "Patient Not Found";
            return null;
        }

        // Get status and core info from localStorage
        const storedPatients = JSON.parse(localStorage.getItem('patients')) || [];
        const patientFromStorage = storedPatients.find(p => p.id === patientId);

        // Combine data from both sources
        const patient = { ...patientDetails, ...(patientFromStorage || {}) };

        document.getElementById('patient-name').textContent = patient.name;
        document.getElementById('patient-id').textContent = `Patient ID: ${patientId}`;
        
        if (patient.dob) {
            const dob = new Date(patient.dob);
            const age = new Date().getFullYear() - dob.getFullYear();
            document.getElementById('patient-dob').textContent = `DOB: ${patient.dob} | Age: ${age}`;
        }

        document.getElementById('patient-gender').textContent = `Gender: ${patient.gender}`;
        
        if (patient.enrollmentDate) {
            document.getElementById('patient-enrollment-date').textContent = `Enrollment Date: ${patient.enrollmentDate}`;
        }
        
        document.getElementById('patient-contact').textContent = `Phone: ${patient.contact}`;
        document.getElementById('patient-address').textContent = `Email: ${patient.address}`;
        document.getElementById('patient-language').textContent = patient.preferredLanguage;
        document.getElementById('patient-nurse').textContent = patient.assignedNurse;

        if (patient.lastCheckin) {
            document.getElementById('last-check-in').textContent = `Last Check-in: ${patient.lastCheckin}`;
        } else {
            const lastBpSampleData = vitalsApp.data['7'];
            const lastBpSys = lastBpSampleData.bp_sys.slice(-1)[0];
            const lastBpDia = lastBpSampleData.bp_dia.slice(-1)[0];
            document.getElementById('last-check-in').textContent = `Last BP: ${lastBpSys}/${lastBpDia}`;
        }

        // Update status indicator
        const statusIndicator = document.getElementById('patient-status-indicator');
        const statusText = document.getElementById('patient-status-text');
        if (statusIndicator && statusText && patient.status) {
            statusText.textContent = `Status: ${patient.status}`;
            statusIndicator.className = `status-indicator ${patient.status.toLowerCase().replace(' ', '-')}`;
        }

        // Update Care Plan Progress Bar
        const progressBar = document.getElementById('care-plan-progress-bar');
        if (progressBar && patient.progress) {
            const progress = patient.progress || 0;
            progressBar.style.width = `${progress}%`;
            progressBar.textContent = `${progress}%`;
        }
        
        return patient;
    }

    // --- SESSION TIMER: Always auto-start, count up from zero on page load ---
    let timerInterval;
    let elapsedSeconds = 0;

    function formatTime(seconds) {
        const h = Math.floor(seconds / 3600).toString().padStart(2, '0');
        const m = Math.floor((seconds % 3600) / 60).toString().padStart(2, '0');
        const s = (seconds % 60).toString().padStart(2, '0');
        return `${h}:${m}:${s}`;
    }

    function updateTimer() {
        elapsedSeconds++;
        const timerDisplay = document.getElementById('timer');
        if (timerDisplay) {
            timerDisplay.textContent = formatTime(elapsedSeconds);
        }
    }

    function startSessionTimer() {
        // Clear any previous timer
        if (timerInterval) clearInterval(timerInterval);
        elapsedSeconds = 0;
        const timerDisplay = document.getElementById('timer');
        if (timerDisplay) timerDisplay.textContent = '00:00:00';
        timerInterval = setInterval(updateTimer, 1000);
    }

    window.addEventListener('load', function() {
        setTimeout(startSessionTimer, 200); // Delay to ensure DOM is ready
    });

    const startButton = document.getElementById('start-session');
    const endSubmitButton = document.getElementById('end-submit');
    const timerDisplay = document.getElementById('timer');
    const timeTrackingModal = document.getElementById('time-tracking-modal');
    const timeTrackingCloseButton = timeTrackingModal.querySelector('.close-button');
    const timeTrackingForm = document.getElementById('time-tracking-form');

    let sessionRunning = false;

    // --- AUTO START SESSION TIMER ON PAGE LOAD ---
    function autoStartSession() {
        // Always clear any previous interval to avoid multiple timers
        if (timerInterval) clearInterval(timerInterval);
        sessionRunning = false;
        elapsedSeconds = 0;
        timerDisplay.textContent = '00:00:00';
        timerInterval = setInterval(updateTimer, 1000);
        sessionRunning = true;
        if (endSubmitButton) endSubmitButton.disabled = false;
    }

    function formatTime(seconds) {
        const h = Math.floor(seconds / 3600).toString().padStart(2, '0');
        const m = Math.floor((seconds % 3600) / 60).toString().padStart(2, '0');
        const s = (seconds % 60).toString().padStart(2, '0');
        return `${h}:${m}:${s}`;
    }

    function updateTimer() {
        elapsedSeconds++;
        timerDisplay.textContent = formatTime(elapsedSeconds);
    }

    function startSession() {
        if (sessionRunning) return;
        timerInterval = setInterval(updateTimer, 1000);
        sessionRunning = true;
        if (startButton) startButton.textContent = 'Pause Session';
        if (endSubmitButton) endSubmitButton.disabled = false;
    }

    function pauseSession() {
        if (!sessionRunning) return;
        clearInterval(timerInterval);
        sessionRunning = false;
        if (startButton) startButton.textContent = 'Resume Session';
    }

    if (startButton) {
        startButton.addEventListener('click', () => {
            if (sessionRunning) {
                pauseSession();
            } else {
                startSession();
            }
        });
    }

    function endSessionAndReset() {
        clearInterval(timerInterval);
        sessionRunning = false;
        elapsedSeconds = 0;
        timerDisplay.textContent = '00:00:00';
        if(startButton) startButton.textContent = 'Start Session';
        if (endSubmitButton) endSubmitButton.disabled = true;
    }

    if (endSubmitButton) {
        endSubmitButton.addEventListener('click', () => {
            timeTrackingModal.style.display = 'block';
        });
    }

    if (timeTrackingCloseButton) {
        timeTrackingCloseButton.onclick = function() {
            timeTrackingModal.style.display = 'none';
        }
    }

    if (timeTrackingForm) {
        timeTrackingForm.addEventListener('submit', (e) => {
            e.preventDefault();

            if (sessionRunning) {
                pauseSession();
            }

            const rpmTime = document.getElementById('rpm-time').value || 0;
            const ccmTime = document.getElementById('ccm-time').value || 0;
            const bhiTime = document.getElementById('bhi-time').value || 0;
            const rtmTime = document.getElementById('rtm-time').value || 0;
            const cgmTime = document.getElementById('cgm-time').value || 0;
            const userName = document.getElementById('user-name-patient').textContent || 'System';
            const notes = document.getElementById('time-tracking-notes').value || '';

            const newRecord = {
                date: new Date().toISOString().slice(0, 10),
                user: userName,
                sessionStart: new Date(Date.now() - elapsedSeconds * 1000),
                sessionEnd: new Date(),
                rpm: parseInt(rpmTime),
                ccm: parseInt(ccmTime),
                bhi: parseInt(bhiTime),
                rtm: parseInt(rtmTime),
                cgm: parseInt(cgmTime),
                totalSession: Math.round(elapsedSeconds / 60),
                notes: notes
            };

            timeTrackingHistory.push(newRecord);
            renderTimeTrackingHistory();

            alert(`Time Logged!`);

            timeTrackingModal.style.display = 'none';
            endSessionAndReset();
            timeTrackingForm.reset();
        });
    }

    // Time Tracking Tab - Manual Entry and Sample Data
    const timeTrackingTable = document.getElementById('time-tracking-history-table');
    const timeTrackingTab = document.getElementById('time-tracking');

    // Sample data for demonstration
    let timeTrackingHistory = [
        {
            date: '2025-07-01', user: 'Nurse Jane', start: '09:00', end: '09:30', rpm: 30, ccm: 10, bhi: 5, rtm: 0, cgm: 0, total: 45, notes: 'Routine check-in.'
        },
        {
            date: '2025-06-30', user: 'Nurse Jane', start: '10:00', end: '10:20', rpm: 20, ccm: 5, bhi: 0, rtm: 0, cgm: 0, total: 25, notes: 'Follow-up call.'
        }
    ];

    function renderTimeTrackingHistory() {
        if (!timeTrackingTable) return;
        const tbody = timeTrackingTable.querySelector('tbody');
        tbody.innerHTML = '';
        timeTrackingHistory.forEach(row => {
            const tr = document.createElement('tr');
            tr.innerHTML = `
                <td>${row.date}</td>
                <td>${row.user}</td>
                <td>${row.start}</td>
                <td>${row.end}</td>
                <td>${row.rpm}</td>
                <td>${row.ccm}</td>
                <td>${row.bhi}</td>
                <td>${row.rtm}</td>
                <td>${row.cgm}</td>
                <td>${row.total}</td>
                <td>${row.notes}</td>
            `;
            tbody.appendChild(tr);
        });
    }

    // Add manual entry form to Time Tracking tab
    if (timeTrackingTab && timeTrackingTable) {
        const form = document.createElement('form');
        form.id = 'manual-time-tracking-form';
        form.innerHTML = `
            <h4>Add Manual Time Tracking Entry</h4>
            <div style="display:flex; flex-wrap:wrap; gap:10px; align-items:center;">
                <input type="date" id="tt-date" required style="width:120px;">
                <input type="text" id="tt-user" placeholder="User" required style="width:110px;">
                <input type="time" id="tt-start" required style="width:100px;">
                <input type="time" id="tt-end" required style="width:100px;">
                <input type="number" id="tt-rpm" placeholder="RPM" min="0" style="width:60px;">
                <input type="number" id="tt-ccm" placeholder="CCM" min="0" style="width:60px;">
                <input type="number" id="tt-bhi" placeholder="BHI" min="0" style="width:60px;">
                <input type="number" id="tt-rtm" placeholder="RTM" min="0" style="width:60px;">
                <input type="number" id="tt-cgm" placeholder="CGM" min="0" style="width:60px;">
                <input type="number" id="tt-total" placeholder="Total" min="0" style="width:70px;">
                <input type="text" id="tt-notes" placeholder="Notes" style="width:160px;">
                <button type="submit">Add</button>
            </div>
        `;
        timeTrackingTab.insertBefore(form, timeTrackingTable);
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            const entry = {
                date: document.getElementById('tt-date').value,
                user: document.getElementById('tt-user').value,
                start: document.getElementById('tt-start').value,
                end: document.getElementById('tt-end').value,
                rpm: parseInt(document.getElementById('tt-rpm').value) || 0,
                ccm: parseInt(document.getElementById('tt-ccm').value) || 0,
                bhi: parseInt(document.getElementById('tt-bhi').value) || 0,
                rtm: parseInt(document.getElementById('tt-rtm').value) || 0,
                cgm: parseInt(document.getElementById('tt-cgm').value) || 0,
                total: parseInt(document.getElementById('tt-total').value) || 0,
                notes: document.getElementById('tt-notes').value
            };
            timeTrackingHistory.unshift(entry);
            renderTimeTrackingHistory();
            form.reset();
        });
        renderTimeTrackingHistory();
    }
    // Vitals & Trends tab functionality
    const vitalsDateRange = document.getElementById('vitals-date-range');
    if (vitalsDateRange) {
        vitalsDateRange.addEventListener('change', () => {
            vitalsApp.renderCharts(vitalsDateRange.value);
        });
    }

    const manualVitalsForm = document.getElementById('manual-vitals-form');
    if (manualVitalsForm) {
        manualVitalsForm.addEventListener('submit', (e) => {
            e.preventDefault();
    
            const date = document.getElementById('vital-date').value;
            const time = document.getElementById('vital-time').value;
            const hr = document.getElementById('vital-hr').value;
            const bpSys = document.getElementById('vital-bp-sys').value;
            const bpDia = document.getElementById('vital-bp-dia').value;
            const spo2 = document.getElementById('vital-spo2').value;
            const weight = document.getElementById('vital-weight').value;
    
            const vitalsLogTable = document.querySelector('#vitals table tbody');
            const newRow = document.createElement('tr');
            newRow.innerHTML = `
                <td>${date}</td>
                <td>${time}</td>
                <td>${hr || 'N/A'}</td>
                <td>${bpSys && bpDia ? `${bpSys}/${bpDia}` : 'N/A'}</td>
                <td>${spo2 || 'N/A'}</td>
                <td>${weight || 'N/A'}</td>
            `;
            vitalsLogTable.prepend(newRow);
    
            // Here you would also update the chart data and re-render the charts
            // For simplicity, this is omitted in this example
    
            manualVitalsForm.reset();
            alert('Manual vitals added successfully!');
        });
    }
    
    const addDiagnosisForm = document.getElementById('add-diagnosis-form');
    if (addDiagnosisForm) {
        addDiagnosisForm.addEventListener('submit', (e) => {
            e.preventDefault();
            const diagnosisName = document.getElementById('diagnosis-name').value;
            const diagnosisCode = document.getElementById('diagnosis-code').value;
    
            if (diagnosisName) {
                const diagnosisList = document.getElementById('diagnosis-list');
                const newListItem = document.createElement('li');
                newListItem.textContent = `${diagnosisName} ${diagnosisCode ? `(${diagnosisCode})` : ''}`;
                diagnosisList.appendChild(newListItem);
    
                addDiagnosisForm.reset();
                alert('Diagnosis added successfully!');
            }
        });
    }
    
    /**
     * A generic function to filter elements based on a date range.
     * @param {object} options - The options for filtering.
     * @param {string} options.buttonId - The ID of the filter button.
     * @param {string} options.fromDateId - The ID of the 'from' date input.
     * @param {string} options.toDateId - The ID of the 'to' date input.
     * @param {string} options.elementSelector - The CSS selector for the elements to filter.
     * @param {function(HTMLElement): string|null} options.dateExtractor - A function to extract the date string (YYYY-MM-DD) from an element.
     */
    function filterElementsByDate(options) {
        const { buttonId, fromDateId, toDateId, elementSelector, dateExtractor } = options;
        const filterBtn = document.getElementById(buttonId);
        if (filterBtn) {
            filterBtn.addEventListener('click', () => {
                const fromDate = document.getElementById(fromDateId).value;
                const toDate = document.getElementById(toDateId).value;
                const elements = document.querySelectorAll(elementSelector);
                elements.forEach(element => {
                    const elementDate = dateExtractor(element);
                    let shouldShow = true;
                    if (elementDate) {
                        if (fromDate && elementDate < fromDate) {
                            shouldShow = false;
                        }
                        if (toDate && elementDate > toDate) {
                            shouldShow = false;
                        }
                    } else {
                        // If there's no date, hide it if a filter is active
                        shouldShow = !(fromDate || toDate);
                    }
                    element.style.display = shouldShow ? '' : 'none';
                });
            });
        }
    }

    // Vitals Log Filter
    filterElementsByDate({
        buttonId: 'vitals-log-filter-btn',
        fromDateId: 'vitals-log-from-date',
        toDateId: 'vitals-log-to-date',
        elementSelector: '#vitals table tbody tr',
        dateExtractor: (row) => row.cells[0] ? row.cells[0].textContent.trim() : null
    });
    
    // Past Appointments Filter
    filterElementsByDate({
        buttonId: 'past-appointment-filter-btn',
        fromDateId: 'past-appointment-from-date',
        toDateId: 'past-appointment-to-date',
        elementSelector: '#past-appointments-table tbody tr',
        dateExtractor: (row) => row.cells[0] ? row.cells[0].textContent.trim() : null
    });
    
    // Documents Filter
    filterElementsByDate({
        buttonId: 'documents-filter-btn',
        fromDateId: 'documents-from-date',
        toDateId: 'documents-to-date',
        elementSelector: '#documents-list li',
        dateExtractor: (item) => {
            const link = item.querySelector('a');
            if (link && link.textContent) {
                const dateMatch = link.textContent.match(/(\d{4}-\d{2}-\d{2})$/);
                return dateMatch ? dateMatch[1] : null;
            }
            return null;
        }
    });
    
    // Clinical Notes Filter
    filterElementsByDate({
        buttonId: 'notes-filter-btn',
        fromDateId: 'notes-from-date',
        toDateId: 'notes-to-date',
        elementSelector: '#clinical-notes-list li',
        dateExtractor: (note) => {
            const noteText = note.textContent;
            const dateMatch = noteText.match(/^(\w+\s+\d{1,2},\s+\d{4})/);
            return dateMatch ? new Date(dateMatch[1]).toISOString().slice(0, 10) : null;
        }
    });
    
    // Patient History Filter
    filterElementsByDate({
        buttonId: 'history-filter-btn',
        fromDateId: 'history-from-date',
        toDateId: 'history-to-date',
        elementSelector: '#patient-history-list li',
        dateExtractor: (item) => item.dataset.date
    });
    
    // Billing Filter
    filterElementsByDate({
        buttonId: 'billing-filter-btn',
        fromDateId: 'billing-from-date',
        toDateId: 'billing-to-date',
        elementSelector: '#billing table tbody tr',
        dateExtractor: (row) => row.cells[1] ? row.cells[1].textContent.trim() : null
    });
    
    // Time Tracking History Filter
    filterElementsByDate({
        buttonId: 'time-tracking-filter-btn',
        fromDateId: 'time-tracking-from-date',
        toDateId: 'time-tracking-to-date',
        elementSelector: '#time-tracking-history-table tbody tr',
        dateExtractor: (row) => row.cells[0] ? row.cells[0].textContent.trim() : null
    });
    
    // Handle Care Plan form submission
    const carePlanForm = document.getElementById('care-plan-form');
    if (carePlanForm) {
        carePlanForm.addEventListener('submit', (e) => {
            e.preventDefault();
            const summary = document.getElementById('care-plan-summary').value;
            const notes = document.getElementById('care-plan-notes').value;
            const actions = document.getElementById('care-plan-actions').value;
    
            // Update the view
            document.getElementById('care-plan-summary-text').textContent = summary;
            document.getElementById('care-plan-notes-text').textContent = notes;
            document.getElementById('care-plan-actions-text').textContent = actions;
    
            // Hide form, show info
            document.getElementById('care-plan-info').classList.remove('hidden');
            document.getElementById('care-plan-form').classList.add('hidden');
    
            alert(`Care Plan Saved!`);
    
            // Update the Key Health Metrics card
            const keyMetricsCarePlan = document.getElementById('key-metrics-care-plan');
            if (keyMetricsCarePlan) {
                keyMetricsCarePlan.textContent = `Care Plan: ${summary}`;
            }
            updatePatientSummary(); // Update summary
        });
    }
    
    // Handle Activity & Lifestyle form submission
    const activityForm = document.getElementById('activity-form');
    if (activityForm) {
        activityForm.addEventListener('submit', (e) => {
            e.preventDefault();
            const steps = document.getElementById('activity-steps').value;
            const sleep = document.getElementById('activity-sleep').value;
            const diet = document.getElementById('activity-diet').value;
            // In a real app, you'd save this data to a backend.
            // For this demo, we'll just show an alert.
            alert(`Activity Log Saved!\nSteps: ${steps}\nSleep: ${sleep}\nDiet: ${diet}`);
        });
    }
    
    // Handle Document Upload form submission
    const documentUploadForm = document.getElementById('document-upload-form');
    if (documentUploadForm) {
        documentUploadForm.addEventListener('submit', (e) => {
            e.preventDefault();
            const fileInput = document.getElementById('document-upload-input');
            const file = fileInput.files[0];
            if (file) {
                const documentsList = document.getElementById('documents-list');
                const newListItem = document.createElement('li');
                const newLink = document.createElement('a');
                newLink.href = '#';
                // Add date to the document name for filtering
                const today = new Date().toISOString().slice(0, 10);
                newLink.textContent = `${file.name} - ${today}`;
                newListItem.appendChild(newLink);
                documentsList.appendChild(newListItem);
                alert(`Document "${file.name}" uploaded successfully!`);
                documentUploadForm.reset();
            }
        });
    }
    
    // Emergency Contacts Tab
    const contactsList = document.getElementById('emergency-contacts-list');
    const addContactBtn = document.getElementById('add-contact-btn');
    const formContainer = document.getElementById('emergency-contact-form-container');
    const contactForm = document.getElementById('emergency-contact-form');
    const cancelFormBtn = document.getElementById('cancel-ec-form-btn');
    const formTitle = document.getElementById('ec-form-title');
    const contactIdInput = document.getElementById('ec-contact-id');

    let emergencyContacts = [
        { id: 1, name: 'Jane Doe', relationship: 'Wife', phone: '(123) 456-7891', date: '2024-07-20' },
        { id: 2, name: 'John Doe Jr.', relationship: 'Son', phone: '(123) 456-7892', date: '2024-07-22' }
    ];

    function renderEmergencyContacts() {
        if (!contactsList) return;
        contactsList.innerHTML = '';
        emergencyContacts
            .sort((a, b) => new Date(b.date) - new Date(a.date))
            .forEach(contact => {
            const div = document.createElement('div');
            div.className = 'emergency-contact-item';
            div.innerHTML = `
                <div class="contact-info">
                    <p><strong>Name:</strong> ${contact.name}</p>
                    <p><strong>Relationship:</strong> ${contact.relationship}</p>
                    <p><strong>Phone:</strong> ${contact.phone}</p>
                    <p><strong>Date:</strong> ${contact.date}</p>
                </div>
                <div class="contact-actions">
                    <button class="btn btn-secondary edit-contact-btn" data-id="${contact.id}">Edit</button>
                    <button class="btn btn-danger delete-contact-btn" data-id="${contact.id}">Delete</button>
                </div>
            `;
            contactsList.appendChild(div);
        });
    }

    function showForm(contact) {
        formTitle.textContent = contact ? 'Edit Emergency Contact' : 'Add Emergency Contact';
        contactIdInput.value = contact ? contact.id : '';
        document.getElementById('ec-name-input').value = contact ? contact.name : '';
        document.getElementById('ec-relationship-input').value = contact ? contact.relationship : '';
        document.getElementById('ec-phone-input').value = contact ? contact.phone : '';
        document.getElementById('ec-date-input').value = contact ? contact.date : '';
        formContainer.classList.remove('hidden');
    }

    function hideForm() {
        formContainer.classList.add('hidden');
        contactForm.reset();
    }

    if (addContactBtn) {
        addContactBtn.addEventListener('click', () => showForm());
    }

    if (cancelFormBtn) {
        cancelFormBtn.addEventListener('click', hideForm);
    }

    if (contactForm) {
        contactForm.addEventListener('submit', (e) => {
            e.preventDefault();
            const id = contactIdInput.value;
            const newContact = {
                name: document.getElementById('ec-name-input').value,
                relationship: document.getElementById('ec-relationship-input').value,
                phone: document.getElementById('ec-phone-input').value,
                date: document.getElementById('ec-date-input').value
            };

            if (id) { // Editing existing contact
                const index = emergencyContacts.findIndex(c => c.id == id);
                emergencyContacts[index] = { ...emergencyContacts[index], ...newContact };
            } else { // Adding new contact
                newContact.id = Date.now(); // simple unique id
                emergencyContacts.push(newContact);
            }

            renderEmergencyContacts();
            hideForm();
        });
    }

    if (contactsList) {
        contactsList.addEventListener('click', (e) => {
            if (e.target.classList.contains('edit-contact-btn')) {
                const id = e.target.dataset.id;
                const contact = emergencyContacts.find(c => c.id == id);
                showForm(contact);
            }
            if (e.target.classList.contains('delete-contact-btn')) {
                const id = e.target.dataset.id;
                if (confirm('Are you sure you want to delete this contact?')) {
                    emergencyContacts = emergencyContacts.filter(c => c.id != id);
                    renderEmergencyContacts();
                }
            }
        });
    }

    // Initial render for emergency contacts
    // renderEmergencyContacts(); // This will be called by initializePatientPage

    // Glucose Chart Modal
    const glucoseChartApp = {
        chart: null,
        data: {
            '7': {
                labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                glucose: [110, 115, 120, 118, 125, 130, 128]
            },
            '30': {
                labels: Array.from({ length: 30 }, (_, i) => `Day ${i + 1}`),
                glucose: Array.from({ length: 30 }, () => Math.floor(Math.random() * 30) + 100)
            },
            'all': {
                labels: Array.from({ length: 90 }, (_, i) => `Day ${i + 1}`),
                glucose: Array.from({ length: 90 }, () => Math.floor(Math.random() * 40) + 90)
            }
        },
        renderChart: function(range = '7') {
            const data = this.data[range];
            if (this.chart) {
                this.chart.destroy();
            }
            const ctx = document.getElementById('glucose-chart').getContext('2d');
            this.chart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: data.labels,
                    datasets: [{
                        label: 'Glucose Levels (mg/dL)',
                        data: data.glucose,
                        borderColor: 'rgba(255, 206, 86, 1)',
                        backgroundColor: 'rgba(255, 206, 86, 0.2)',
                        borderWidth: 2,
                        pointRadius: 5,
                        pointHoverRadius: 7
                    }]
                },
                options: {
                    scales: {
                        y: {
                            beginAtZero: false
                        }
                    }
                }
            });
        }
    };

    const viewChartLink = document.getElementById('view-chart-link');
    const chartModal = document.getElementById('chart-modal');
    const closeButton = chartModal.querySelector('.close-button');
    const chartDateRange = document.getElementById('chart-date-range');

    if (viewChartLink) {
        viewChartLink.addEventListener('click', (e) => {
            e.preventDefault();
            chartModal.style.display = 'block';
            glucoseChartApp.renderChart(chartDateRange.value);
        });
    }

    if (closeButton) {
        closeButton.onclick = function() {
            chartModal.style.display = 'none';
        }
    }

    if (chartDateRange) {
        chartDateRange.addEventListener('change', () => {
            glucoseChartApp.renderChart(chartDateRange.value);
        });
    }

    window.onclick = function(event) {
        if (chartModal && event.target == chartModal) {
            chartModal.style.display = 'none';
        }
        if (timeTrackingModal && event.target == timeTrackingModal) {
            timeTrackingModal.style.display = 'none';
        }
    }

    // Run patient-specific initialization only on the patient page
    if (document.getElementById('patient-name')) {
        initializePatientPage();
    }
});