document.addEventListener('DOMContentLoaded', () => {
    const themeSelect = document.getElementById('theme-select');
    const body = document.body;

    function applyTheme(theme) {
        body.className = '';
        if (theme !== 'light') {
            body.classList.add(theme);
        }
        localStorage.setItem('selectedTheme', theme);
    }

    if(themeSelect) {
        themeSelect.addEventListener('change', () => {
            applyTheme(themeSelect.value);
        });

        const savedTheme = localStorage.getItem('selectedTheme');
        if (savedTheme) {
            themeSelect.value = savedTheme;
            applyTheme(savedTheme);
        } else {
            applyTheme('purple');
            themeSelect.value = 'purple';
        }
    }

    const patientTableBody = document.querySelector('#patient-table tbody');
    const searchInput = document.getElementById('search-patient');
    const statusFilter = document.getElementById('status-filter');
    const triageFilter = document.getElementById('triage-filter');
    const alertsContainer = document.getElementById('alerts-container');
    const notificationPopup = document.getElementById('notification-popup');


    // Modal elements
    const modal = document.getElementById('create-patient-modal');
    const createPatientBtn = document.getElementById('create-patient-btn');
    const closeButton = document.querySelector('.close-button');
    const createPatientForm = document.getElementById('create-patient-form');

    let patients = [];
    const PATIENT_DATA_VERSION = '1.8'; // Increment this version to force a data refresh

    // Use localStorage as a simple database to keep patient data in sync
    function initializePatients() {
        const storedVersion = localStorage.getItem('patientDataVersion');
        const storedPatients = localStorage.getItem('patients');

        if (storedPatients && storedVersion === PATIENT_DATA_VERSION) {
            try {
                patients = JSON.parse(storedPatients);
                // Ensure age and lastBP are always present and correct
                patients.forEach(p => {
                    if (p.dob && (!p.age || isNaN(p.age))) {
                        const dobDate = new Date(p.dob);
                        const today = new Date();
                        let age = today.getFullYear() - dobDate.getFullYear();
                        const m = today.getMonth() - dobDate.getMonth();
                        if (m < 0 || (m === 0 && today.getDate() < dobDate.getDate())) {
                            age--;
                        }
                        p.age = age;
                    }
                    if (!p.lastBP) {
                        p.lastBP = 'N/A';
                    }
                });
                localStorage.setItem('patients', JSON.stringify(patients));
                return;
            } catch (e) {
                console.error("Error parsing patients from localStorage", e);
            }
        }
        
        // If we reach here, it means localStorage is empty, invalid, or has old data.
        // So, we re-initialize with the hardcoded list.
        patients = [
            { name: 'Jane Smith', id: '789012', dob: '1965-03-12', deviceType: 'BP Monitor', age: 59, status: 'Critical', lastCheckin: '15 mins ago', lastBP: '1 day ago', triage: 'routine', progress: 90, enrollmentDate: '2025-06-01', commCount: 12, measureDays: 25, rpmMin: 10, ccmMin: 45, bhiMin: 20, rtmMin: 15, cgmMin: 110, lastBillingDate: '2025-06-01', notifications: 2, myPatient: true, providerGoal: 20 },
            { name: 'Bob Johnson', id: '345678', dob: '1958-07-24', deviceType: 'Glucometer', age: 65, status: 'At Risk', lastCheckin: '1 hour ago', lastBP: '2 days ago', triage: 'urgent', progress: 70, enrollmentDate: '2025-05-20', commCount: 5, measureDays: 20, rpmMin: 40, ccmMin: 20, bhiMin: 10, rtmMin: 5, cgmMin: 70, lastBillingDate: '2025-05-20', notifications: 1, myPatient: true, providerGoal: 40 },
            { name: 'Sarah Connor', id: '654321', dob: '1984-05-01', deviceType: 'Pulse Oximeter', age: 40, status: 'At Risk', lastCheckin: '30 mins ago', lastBP: '1 day ago', triage: 'urgent', progress: 60, enrollmentDate: '2025-06-10', commCount: 8, measureDays: 22, rpmMin: 60, ccmMin: 30, bhiMin: 12, rtmMin: 8, cgmMin: 85, lastBillingDate: '2025-06-10', notifications: 4, myPatient: false, providerGoal: 60 },
            { name: 'John Doe', id: '123456', dob: '1970-01-15', deviceType: 'Weight Scale', age: 54, status: 'Stable', lastCheckin: '2 hours ago', lastBP: '3 days ago', triage: 'routine', progress: 85, enrollmentDate: '2025-04-15', commCount: 8, measureDays: 25, rpmMin: 15, ccmMin: 30, bhiMin: 15, rtmMin: 10, cgmMin: 90, lastBillingDate: '2025-06-15', notifications: 0, myPatient: true, providerGoal: 20 },
            { name: 'Alice Williams', id: '901234', dob: '1990-11-30', deviceType: 'BP Monitor', age: 33, status: 'Stable', lastCheckin: '3 hours ago', lastBP: '4 days ago', triage: 'routine', progress: 95, enrollmentDate: '2025-05-01', commCount: 10, measureDays: 28, rpmMin: 20, ccmMin: 35, bhiMin: 18, rtmMin: 12, cgmMin: 100, lastBillingDate: '2025-06-01', notifications: 2, myPatient: false, providerGoal: 40 },
            { name: 'Charlie Brown', id: '567890', dob: '2005-08-17', deviceType: 'Glucometer', age: 18, status: 'Stable', lastCheckin: '5 hours ago', lastBP: '5 days ago', triage: 'routine', progress: 80, enrollmentDate: '2025-03-30', commCount: 7, measureDays: 26, rpmMin: 55, ccmMin: 25, bhiMin: 14, rtmMin: 9, cgmMin: 80, lastBillingDate: '2025-05-30', notifications: 0, myPatient: true, providerGoal: 60 },
        ];
        localStorage.setItem('patients', JSON.stringify(patients));
        localStorage.setItem('patientDataVersion', PATIENT_DATA_VERSION);
    }

    function renderAlerts() {
        const criticalPatients = patients.filter(p => p.status === 'Critical');
        if (criticalPatients.length > 0) {
            alertsContainer.innerHTML = '<h2>Alerts</h2>';
            criticalPatients.forEach(p => {
                const alertDiv = document.createElement('div');
                alertDiv.className = 'alert';
                alertDiv.innerHTML = `<strong>${p.name} (ID: ${p.id})</strong> has a critical status. Please review immediately.`;
                alertsContainer.appendChild(alertDiv);
            });
        } else {
            alertsContainer.innerHTML = '<h2>Alerts</h2><p>No critical alerts at this time.</p>';
        }
    }

    function renderPatients(filteredPatients) {
        patientTableBody.innerHTML = '';
        const patientsToRender = filteredPatients || patients;

        patientsToRender.forEach(patient => {
            const row = document.createElement('tr');
            row.className = `status-${patient.status.toLowerCase().replace(' ', '-')}`;

            // Name (link)
            const nameCell = document.createElement('td');
            const link = document.createElement('a');
            link.href = `patient.html?patientId=${patient.id}`;
            link.textContent = patient.name;
            nameCell.appendChild(link);

            // Add status-indicator and urgent class for urgent triage
            if (patient.triage === 'urgent') {
                row.classList.add('status-indicator', 'urgent');
                nameCell.style.color = '#ff8c00';
                row.style.color = '#ff8c00'; // Force all text in row to be amber for urgent
            }

            // DOB
            const dobCell = document.createElement('td');
            dobCell.textContent = patient.dob;

            // Device Type
            const deviceTypeCell = document.createElement('td');
            deviceTypeCell.textContent = patient.deviceType || 'N/A';

            // Last Reading (was Last BP)
            const lastReadingCell = document.createElement('td');
            lastReadingCell.textContent = patient.lastBP;

            // Last Note entered (was Last billing date)
            const lastNoteCell = document.createElement('td');
            lastNoteCell.textContent = patient.lastBillingDate;

            // Comm. Count
            const commCountCell = document.createElement('td');
            commCountCell.textContent = patient.commCount;

            // Measure Days
            const measureDaysCell = document.createElement('td');
            measureDaysCell.textContent = patient.measureDays;

            // RPM
            const rpmMinCell = document.createElement('td');
            rpmMinCell.innerHTML = `
                <div class="minute-progress-bar">
                    <div class="minute-bar bar1" style="width: ${Math.min(100, (patient.rpmMin || 0) * 2)}%"></div>
                    <div class="minute-bar bar2" style="width: ${Math.min(100, (patient.rpmMin || 0) * 1.2)}%"></div>
                </div>
            `;

            // CCM
            const ccmMinCell = document.createElement('td');
            ccmMinCell.innerHTML = `
                <div class="minute-progress-bar">
                    <div class="minute-bar bar1" style="width: ${Math.min(100, (patient.ccmMin || 0) * 2)}%"></div>
                    <div class="minute-bar bar2" style="width: ${Math.min(100, (patient.ccmMin || 0) * 1.2)}%"></div>
                </div>
            `;

            // BHI
            const bhiMinCell = document.createElement('td');
            bhiMinCell.innerHTML = `
                <div class="minute-progress-bar">
                    <div class="minute-bar bar1" style="width: ${Math.min(100, (patient.bhiMin || 0) * 2)}%"></div>
                    <div class="minute-bar bar2" style="width: ${Math.min(100, (patient.bhiMin || 0) * 1.2)}%"></div>
                </div>
            `;

            // RTM
            const rtmMinCell = document.createElement('td');
            rtmMinCell.innerHTML = `
                <div class="minute-progress-bar">
                    <div class="minute-bar bar1" style="width: ${Math.min(100, (patient.rtmMin || 0) * 2)}%"></div>
                    <div class="minute-bar bar2" style="width: ${Math.min(100, (patient.rtmMin || 0) * 1.2)}%"></div>
                </div>
            `;

            // CGM
            const cgmMinCell = document.createElement('td');
            cgmMinCell.innerHTML = `
                <div class="minute-progress-bar">
                    <div class="minute-bar bar1" style="width: ${Math.min(100, (patient.cgmMin || 0) * 2)}%"></div>
                    <div class="minute-bar bar2" style="width: ${Math.min(100, (patient.cgmMin || 0) * 1.2)}%"></div>
                </div>
            `;

            // Provider Minute Goals progress bar
            const providerMinuteGoalsCell = document.createElement('td');
            // Use patient.providerGoal if present, otherwise default to 40
            const goal = patient.providerGoal || 40;
            const value = Math.min(patient.rpmMin || 0, goal);
            const percent = Math.round((value / goal) * 100);
            providerMinuteGoalsCell.innerHTML = `
                <div class="progress-bar-container">
                    <div class="progress-bar" style="width: ${percent}%; background-color: #007bff; min-width: 8px;">${percent}%</div>
                    <div class="provider-minute-goal-label">${value} / ${goal} min</div>
                </div>
            `;

            // Notifications
            const notificationsCell = document.createElement('td');
            if (patient.notifications > 0) {
                notificationsCell.classList.add('has-notifications');
                const notificationIcon = document.createElement('span');
                notificationIcon.className = 'notification-icon';
                notificationIcon.textContent = patient.notifications;
                notificationsCell.appendChild(notificationIcon);

                notificationsCell.addEventListener('mouseenter', (e) => {
                    const notificationDetails = [
                        `Medication reminder for Aspirin.`,
                        `Blood pressure reading is high.`,
                        `Appointment scheduled for tomorrow.`,
                        `New document available for review.`
                    ].slice(0, patient.notifications).map(detail => `<div>${detail}</div>`).join('');

                    notificationPopup.innerHTML = notificationDetails;
                    notificationPopup.style.display = 'block';
                    const rect = notificationIcon.getBoundingClientRect();
                    notificationPopup.style.left = `${rect.left + window.scrollX - (notificationPopup.offsetWidth / 2) + (rect.width / 2)}px`;
                    notificationPopup.style.top = `${rect.bottom + window.scrollY + 5}px`;
                });

                notificationsCell.addEventListener('mouseleave', () => {
                    notificationPopup.style.display = 'none';
                });
            }

            // Append all cells in the new order (without myPatientCell)
            row.appendChild(nameCell);
            row.appendChild(dobCell);
            row.appendChild(deviceTypeCell);
            row.appendChild(lastReadingCell);
            row.appendChild(lastNoteCell);
            row.appendChild(commCountCell);
            row.appendChild(measureDaysCell);
            row.appendChild(rpmMinCell);
            row.appendChild(ccmMinCell);
            row.appendChild(bhiMinCell);
            row.appendChild(rtmMinCell);
            row.appendChild(cgmMinCell);
            row.appendChild(providerMinuteGoalsCell);
            row.appendChild(notificationsCell);

            patientTableBody.appendChild(row);
        });
    }

    function sortPatients(key, order) {
        const statusOrder = { 'Critical': 1, 'At Risk': 2, 'Stable': 3 };

        // Sort the array of all patients
        patients.sort((a, b) => {
            let valA, valB;

            // Handle different keys
            if (key === 'status') {
                valA = statusOrder[a.status];
                valB = statusOrder[b.status];
            } else if (key === 'lastBP') {
                const extractDays = (str) => {
                    if (!str) return Infinity;
                    const match = str.match(/^(\d+)/);
                    return match ? parseInt(match[1], 10) : Infinity;
                };
                valA = extractDays(a.lastBP);
                valB = extractDays(b.lastBP);
            } else if (key === 'lastCheckin') {
                const parseTimeAgo = (str) => {
                    if (!str) return Infinity;
                    if (str.toLowerCase() === 'just now') return 0;
                    const match = str.match(/^(\d+)\s+(min|mins|hour|hours)/i);
                    if (match) {
                        const value = parseInt(match[1], 10);
                        const unit = match[2].toLowerCase();
                        if (unit.startsWith('min')) {
                            return value; // Time in minutes
                        } else if (unit.startsWith('hour')) {
                            return value * 60; // Convert hours to minutes
                        }
                    }
                    return Infinity; // Fallback for unparsed strings
                };
                valA = parseTimeAgo(a.lastCheckin);
                valB = parseTimeAgo(b.lastCheckin);
            } else {
                valA = a[key];
                valB = b[key];
            }

            // Comparison logic
            if (valA === null || valA === undefined) valA = key === 'name' ? '' : Infinity;
            if (valB === null || valB === undefined) valB = key === 'name' ? '' : Infinity;


            if (typeof valA === 'string' && typeof valB === 'string') {
                return order === 'asc' ? valA.localeCompare(valB) : valB.localeCompare(valA);
            } else {
                if (valA < valB) return order === 'asc' ? -1 : 1;
                if (valA > valB) return order === 'asc' ? 1 : -1;
                return 0;
            }
        });

        // After sorting, re-apply the current filters and render the table
        filterAndRender();
    }

    document.querySelectorAll('#patient-table th').forEach(header => {
        header.addEventListener('click', () => {
            let sortKey = header.dataset.sortKey;
            // Map display keys to actual patient object keys
            if (sortKey === 'lastReading') sortKey = 'lastBP';
            if (sortKey === 'lastNote') sortKey = 'lastBillingDate';
            if (!sortKey) return;

            const currentOrder = header.classList.contains('sort-asc') ? 'desc' : 'asc';

            document.querySelectorAll('#patient-table th').forEach(th => {
                th.classList.remove('sort-asc', 'sort-desc');
            });

            header.classList.add(currentOrder === 'asc' ? 'sort-asc' : 'sort-desc');

            sortPatients(sortKey, currentOrder);
        });
    });

    function filterAndRender() {
        const searchTerm = searchInput.value.toLowerCase();
        const status = statusFilter.value;
        const triage = triageFilter.value;
        // Removed showMyPatients and myPatientMatch logic

        const filtered = patients.filter(patient => {
            const nameMatch = patient.name.toLowerCase().includes(searchTerm);
            const idMatch = patient.id.includes(searchTerm);
            const statusMatch = status === 'all' || patient.status === status;
            const triageMatch = triage === 'all' || patient.triage === triage;
            // Only filter by name, id, status, and triage
            return (nameMatch || idMatch) && statusMatch && triageMatch;
        });

        renderPatients(filtered);
    }

    // Modal event listeners
    createPatientBtn.onclick = function() {
        modal.style.display = 'block';
    }

    closeButton.onclick = function() {
        modal.style.display = 'none';
    }

    window.onclick = function(event) {
        if (event.target == modal) {
            modal.style.display = 'none';
        }
    }

    createPatientForm.addEventListener('submit', (e) => {
        e.preventDefault();
        const dob = document.getElementById('patient-dob-input')?.value || '';
        let age = '';
        if (dob) {
            const dobDate = new Date(dob);
            const today = new Date();
            age = today.getFullYear() - dobDate.getFullYear();
            const m = today.getMonth() - dobDate.getMonth();
            if (m < 0 || (m === 0 && today.getDate() < dobDate.getDate())) {
                age--;
            }
        }
        const newPatient = {
            name: document.getElementById('patient-name').value,
            id: document.getElementById('patient-id').value,
            dob: dob,
            age: age,
            status: document.getElementById('patient-status').value,
            lastCheckin: 'Just now',
            lastBP: 'N/A',
            triage: 'routine',
            progress: 0,
            enrollmentDate: new Date().toISOString().split('T')[0],
            commCount: 0,
            measureDays: 0,
            rpmMin: 0,
            ccmMin: 0,
            bhiMin: 0,
            rtmMin: 0,
            cgmMin: 0,
            lastBillingDate: new Date().toISOString().split('T')[0],
            notifications: 0,
            myPatient: false
        };
        patients.unshift(newPatient);
        localStorage.setItem('patients', JSON.stringify(patients));
        filterPatients();
        renderAlerts();
        modal.style.display = 'none';
        createPatientForm.reset();
    });

    searchInput.addEventListener('input', filterAndRender);
    statusFilter.addEventListener('change', filterAndRender);
    triageFilter.addEventListener('change', filterAndRender);
    // Remove myPatientsFilter event listener
    // myPatientsFilter.addEventListener('change', filterAndRender);

    // Set user name
    const userNameSpan = document.getElementById('user-name');
    if (userNameSpan) {
        // In a real application, you would get this from a login system.
        userNameSpan.textContent = 'Milan Nesic'; 
    }

    // Initial Load
    initializePatients();
    // Set the initial sort order by status (asc) and update the UI to reflect this
    sortPatients('status', 'asc');
    const initialSortHeader = document.querySelector('th[data-sort-key="status"]');
    if(initialSortHeader) {
        initialSortHeader.classList.add('sort-asc');
    }
    renderAlerts();
    renderPatients(); // Ensure patient data is rendered on load
});
